/**
* \file: AoapTransport.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Baidu CarLife
*
* \author: P. Govindaraju / RBEB/GM / Pradeepa.Govindaraju@in.bosch.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

/* *************  includes  ************* */
#include <AoapTransport.h>


#include <adit_logging.h>
#include <pthread_adit.h>
#include <adit_typedef.h>
#include <sys_time_adit.h>
#include <stdint.h>
#include <stdio.h>

#include <sys/unistd.h>
#include <sys/types.h>

#include <fcntl.h>

#include <assert.h>

#include <aoap_types.h>
#include <aoap.h>

using namespace std;

/* *************  functions  ************* */
LOG_IMPORT_CONTEXT(bdcl_transport)

namespace adit { namespace bdcl {


AoapDevice::AoapDevice(){
    mSwitchResult = -1;
    sem_init(&mAoapSyncSem, 0, 0);
};

int AoapDevice::switchDevice(aoapDeviceInfo_t* inAoapDeviceInfo, aoapTransportInfo_t* outTransportInfo, unsigned int inTimeout) {
    int ret = 0;

    if (outTransportInfo == nullptr || inAoapDeviceInfo == nullptr) {
        LOG_ERROR((bdcl_transport, "%s() Invalid parameter outTransportInfo=%p, inAoapDeviceInfo=%p",
                __FUNCTION__, outTransportInfo, inAoapDeviceInfo));
        return -1;
    }

    /* create SyncContext. will be deleted by itself */

    /* create AOAP library accessory session */
    outTransportInfo->aoapAccessoryId = aoap_create_accessory((t_aoap_accessory_param*)&inAoapDeviceInfo->aoapAccessoryInfo);
    if (outTransportInfo->aoapAccessoryId < 0) {
        LOG_ERROR((bdcl_transport, "%s() Create libaoap accessory session failed=%d",
                    __FUNCTION__, outTransportInfo->aoapAccessoryId));
        ret = -1;
    } else {
        if (0 == inTimeout) {
            LOG_WARN((bdcl_transport, "%s() timeout is=%d. Use timeout > 0.", __FUNCTION__, inTimeout));
            inTimeout = 8;
            LOG_WARN((bdcl_transport, "%s() Adjust timeout to %d", __FUNCTION__, inTimeout));
        }
        /* change connect timeout based on the Application setting */
        aoap_set_connect_timeout(outTransportInfo->aoapAccessoryId, inTimeout / 1000);

        /* switch to accessory mode */
        outTransportInfo->aoapDeviceId = aoap_connect(outTransportInfo->aoapAccessoryId, \
                                                      inAoapDeviceInfo->vendorId, inAoapDeviceInfo->productId, inAoapDeviceInfo->pSerial, \
                                                      &switchCallback, inAoapDeviceInfo->aoapAccessoryInfo.enableAudio, this);
        if (0 > outTransportInfo->aoapDeviceId) {
            LOG_ERROR((bdcl_transport, "%s() Create libaoap connection failed=%d", __FUNCTION__, outTransportInfo->aoapDeviceId));
            ret = -1;
        } else {
            /* switch to accessory mode initiated.
             * wait for AOAP library switch callback (condition to become true) */
            if (0 > sem_wait(&mAoapSyncSem)) {
                LOG_ERROR((bdcl_transport, "%s() sem_wait() failed", __FUNCTION__));
            }
            if (0 == mSwitchResult) {
                LOG_INFO((bdcl_transport, "%s() Device with serial=%s switched to accessory mode =%d",
                        __FUNCTION__, inAoapDeviceInfo->pSerial, mSwitchResult));
                /* return switch result */
                ret = mSwitchResult;
            } else if (AOAP_ERROR_ALREADY_DONE == mSwitchResult) {
                LOG_INFO((bdcl_transport, "%s() Device with serial=%s re-connected to accessory, aoapDeviceId =%d",
                        __FUNCTION__, inAoapDeviceInfo->pSerial, outTransportInfo->aoapDeviceId));
                ret = 0;
            } else {
                LOG_ERROR((bdcl_transport, "%s() Switch device with serial=%s to accessory mode failed =%d",
                        __FUNCTION__, inAoapDeviceInfo->pSerial, mSwitchResult));
                ret = mSwitchResult;
            }
        }
        /* de-initialize AOAP library in case switch failed */
        if (0 > ret) {
            LOG_INFO((bdcl_transport, "%s() Cleanup AOAP library accessory session due to switch device failed.", __FUNCTION__));

            /* cleanup AOAP library accessory session
             * if no devices associated to the accessory session. */
            aoap_defer_delete_accessory(outTransportInfo->aoapAccessoryId, outTransportInfo->aoapDeviceId);
            /* did not set aoapAccessoryId to -1 to support multiple AAP devices */
            outTransportInfo->aoapDeviceId = -1;
        }
    }

    return ret;
}

void AoapDevice::switchCallback(int accessoryId, int deviceId, int result, void *pToken, unsigned int audioSupport) {
    accessoryId = accessoryId;
    deviceId = deviceId;
    audioSupport = audioSupport;

    if (AOAP_ERROR_ALREADY_DONE == result) {
        LOG_INFO((bdcl_transport, "%s() Switch to accessory mode already done=%d", __FUNCTION__, result));
        /* Note: If result is AOAP_ERROR_ALREADY_DONE
         * Expect that callback was triggered by API aoap_connect()->connectDevice()
         * and aoap_connect() return in switchDevice() with valid value.
         * In this case, do not use SyncContext for synchronization! */
    } else if (0 > result) {
        LOG_ERROR((bdcl_transport, "%s() Switch to accessory mode failed=%d", __FUNCTION__, result));
    } else {
        LOG_INFO((bdcl_transport, "%s() Switch to accessory mode success=%d", __FUNCTION__, result));
    }
    if (NULL != pToken) {
        AoapDevice* me = (AoapDevice*)pToken;
        me->mSwitchResult = result;
        if (0 > sem_post(&me->mAoapSyncSem)) {
            LOG_ERROR((bdcl_transport, "%s() sem_post() failed", __FUNCTION__));
        }
    } else {
        LOG_ERROR((bdcl_transport, "%s() pToken is NULL. Cannot signal condition", __FUNCTION__));
        assert(pToken);
    }
}


} } /* namespace adit { namespace bdcl { */



